<?php
/* --------------------------------------------------------------
 AuthMiddleware.php 2020-02-14
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Application\Middleware;

use Gambio\Admin\Customer\SimpleCustomerService;
use Gambio\Core\Application\ValueObjects\Url;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\RequestHandlerInterface;
use Slim\Http\Factory\DecoratedResponseFactory;
use Slim\Http\Response;
use Slim\Psr7\Factory\ResponseFactory;
use Slim\Psr7\Factory\StreamFactory;

/**
 * Class AuthMiddleware
 * @package Gambio\Admin\Application\Middleware
 */
class AuthMiddleware
{
    /**
     * @var SimpleCustomerService
     */
    private $customerService;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Url
     */
    private $url;
    
    
    /**
     * AuthMiddleware constructor.
     *
     * @param SimpleCustomerService $customerService
     * @param UserPreferences       $userPreferences
     * @param Url                   $url
     */
    public function __construct(SimpleCustomerService $customerService, UserPreferences $userPreferences, Url $url)
    {
        $this->customerService = $customerService;
        $this->userPreferences = $userPreferences;
        $this->url             = $url;
    }
    
    
    /**
     * Middleware callback ensuring that a admin is logged in.
     *
     * @param ServerRequestInterface  $request
     * @param RequestHandlerInterface $handler
     *
     * @return ResponseInterface
     */
    public function __invoke(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        if (!$this->userPreferences->isCustomer()
            || !$this->customerService->isAdmin($this->userPreferences->customerId())) {
            return $this->redirectToLoginResponse();
        }
        
        return $handler->handle($request);
    }
    
    
    /**
     * Creates a response that is redirecting the the shop store's login.php page.
     *
     * @return Response
     */
    private function redirectToLoginResponse(): Response
    {
        $responseFactory = (new DecoratedResponseFactory(new ResponseFactory(), new StreamFactory()));
        $response        = $responseFactory->createResponse(401);
        
        return $response->withRedirect("{$this->url->base()}/login.php");
    }
}